package service;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;
import org.jgrapht.graph.*;


public class TransitionSystemImpl<S extends State,A extends Transition> extends TransitionSystem<S,A> {
    private DirectedMultigraph<S, Transition> directedGraph=new DirectedMultigraph<S,Transition>(Transition.class);
    private String name,errorStr;
    private S initialState;
    private int numStates=0;
    
    public TransitionSystemImpl() {
    	name="";
    	initialState=null;
	}

	/**
	 * Add a new action to the transition system
	 * @param stateFrom the source state of the action 
	 * @param stateTo the target state of the action
	 * @param a the action that 
	 * @throws Exception return a Exception if the argument state is null of if the stateFrom or stateTo don't exist in the transition system
	 */
	public void addAction(S stateFrom, S stateTo, A a) throws Exception{
		if (stateFrom==null)
			throw new Exception("the argument stateFrom is null");
		if (stateTo==null)
			throw new Exception("the argument stateTo is null");
		if (a==null)
			throw new Exception("the argument a is null");
		directedGraph.addEdge(stateFrom, stateTo, a);
	}

	/**
	 * Add a new state to the transition system
	 * @param state the state that we want add to the transition system
	 * @throws Exception return a Exception if the argument state is null or if there is already a state with the same name in the transition system or if the state is initial and there is already a initial state in the transistion system
	 */
	public void addState(S state) throws InvalidArgumentException, DuplicateStateException {
		if (state==null){
			throw new InvalidArgumentException("the argument state is null");
		}else if (directedGraph.containsVertex(state)){
			throw new DuplicateStateException("the state with the name \""+state.getName()+"\" already exists.");
		}else if (state.isInitial() && initialState != null)
			throw new InvalidArgumentException("there is already a initial state with the name: "+state.getName());

		if(state.isInitial())
			initialState=state;
		directedGraph.addVertex(state);
		numStates++;
	}

	@SuppressWarnings("unchecked")
	/**
	 *	Returns a Set that contain all the actions contained in the transistion system that connects the state "from" with the state "to" 
	 * @return the Set described above or a empty set
	 * @throws Exception return a Exception if the argument "from" or "to" are null or they aren't contained in the transition sytem
	 */
	public Set<A> getActions(S from, S to) throws Exception {
		if (from==null)
			throw new Exception("the argument from is null");
		if (to==null)
			throw new Exception("the argument to is null");
		HashSet<Transition> temp=new HashSet(directedGraph.outgoingEdgesOf(from));
		HashSet<Transition> result=new HashSet();
		Iterator<Transition> it=temp.iterator();
		while (it.hasNext()){
			Transition a=it.next();
			if(directedGraph.getEdgeTarget(a).equals(to))
			{
				result.add(a);
			}
		}		
		return (Set<A>)result;
	}

	@SuppressWarnings("unchecked")
	/**
	 *	Returns a Set that contain all the actions contained in the transistion system that have the source state "from" 
	 * @return the Set described above or a empty set
	 * @throws Exception return a Exception if the argument "from" is null or they aren't contained in the transition sytem
	 */
	public Set<A> getActionsOf(S from) throws Exception {
		if (from==null)
			throw new Exception("the argument from is null");
		return new HashSet<A>((Set<A>)directedGraph.outgoingEdgesOf(from));
	}

	@SuppressWarnings("unchecked")
	/**
	 *	Returns a Set that contain all the actions of the transition system 
	 * @return the Set described above
	 */
	public Set<A> getAllActions() {
		return new HashSet((Set<A>)directedGraph.edgeSet());
	}

	/**
	 *	Returns a Set that contain all the states of the transition system 
	 * @return the Set described above
	 */
	public Set<S> getAllStates() {
		return new HashSet<S>(directedGraph.vertexSet());
	}

	/**
	 * Return the name associated to the transition system
	 * @return the string contains the name of the transition system
	 */
	public String getName() {
		return name;
	}

	/**
	 * Return the source state of the action used as argument 
	 * @throws Exception This method throws a exception if the argument action is null or if the transition system doesn't contain the action
	 */
	public S getSourceState(A action) throws Exception{
		if (action==null)
			throw new Exception("the argument action is null");
		if(directedGraph.containsEdge((Transition)action))
			return directedGraph.getEdgeSource((Transition)action);
		else
			throw new Exception("The action isn't cointained in the graph");
	}

	/**
	 * Return the state identified by the name passed as argument
	 * @param name the name of the state that we want search
	 * @throws Exception This method throws a exception if the argument name is null or if the transition system doesn't contain the action
	 */
	public S getState(String name) {
		
		//TODO da modificare solo soluzione temporanea
		Set<S> set=directedGraph.vertexSet();
		Iterator<S> it=(Iterator<S>)set.iterator();
		while(it.hasNext()){
			S s=it.next();
			if(s.getName().equals(name))
				return s;
		}
		return null;
	}

	/**
	 * Return the target state of the action used as argument 
	 * @throws Exception This method throws a exception if the argument action is null or if the transition system doesn't contain the action
	 */
	public S getTargetState(A action) throws Exception {
		if (action==null)
			throw new Exception("the argument action is null");
		if(directedGraph.containsEdge((Transition)action))
			return directedGraph.getEdgeTarget((Transition)action);
		else
			throw new Exception("The action isn't cointained in the graph");
	}


	/**
	 * Set the name associated to the transition system
	 * @param name the new name of the transition system
	 * @throws Exception return a Exception if the string name is null
	 */
	public void setName(String name) throws Exception {
		if (name!=null)
			this.name=name;
		else throw new Exception("the argument name is null");
	}

	/**
	 *	Returns if the transition system represented is valid or not 
	 *  To check the validity of the transistion system this method check if:
	 *  - there is a unique initial state
	 *  - there is one or more final state
	 *  - all the states are reachable from the initial state 
	 * @return return true if the transition system is valid else return not
	 */
	public boolean isValid() {
		//there is a unique initial state?
		{
			if(initialState == null){
			errorStr="There isn't a initial state";
			return false;
			}
		}
		//there is one or more final states?
		{
			Set<S> states=getAllStates();
			Iterator<S> it=states.iterator();
			State state=null;
			boolean tsHasFinal=false;
			while(it.hasNext()){
				state=it.next();
				if(state.isFinal()){
					tsHasFinal=true;
					break;
				}
			}
			if(!tsHasFinal){
				errorStr="There isn't one or more final state";
				return false;
			}
		}
		//check if all the states are reachable from the initial state 
		{
			HashSet<S> markedState=new HashSet<S>();
			HashSet<S> notVisitedState=new HashSet<S>();
			try {
				markedState.add(initialState);
				Set<A> actionFrom=getActionsOf(initialState);
				Iterator<A> it=actionFrom.iterator();
				while(it.hasNext()){
					A action=it.next();
					S targetState=getTargetState(action);
					if (!markedState.contains(targetState))
						notVisitedState.add(targetState);
				}
				while(!notVisitedState.isEmpty()){
					S visitedState=notVisitedState.iterator().next();
					notVisitedState.remove(visitedState);
					markedState.add(visitedState);
					
					actionFrom=getActionsOf(visitedState);
					it=actionFrom.iterator();
	
					while(it.hasNext()){
						A action=it.next();
						
			
						S targetState=getTargetState(action);
						if (!markedState.contains(targetState))
							notVisitedState.add(targetState);
					}
				}
			}catch(Exception e){
				e.printStackTrace();
			}
			if(markedState.size()!=numStates){
				Set<S> states=getAllStates();
				states.removeAll(markedState);
				String stateUnreachableStr="";
				Iterator<S> it=states.iterator();
				while(it.hasNext()){
					stateUnreachableStr=stateUnreachableStr+it.next().getName()+",";
				}
				errorStr="Some states are unreachable: "+stateUnreachableStr;
				errorStr=errorStr.substring(0,errorStr.length()-1)+".";
				return false;
			}
		}
		return true;
	}

	
	/**
	 * If the transition system is invalid return a string that contain the cause of non-validity
	 * else return null
	 */
	public String getError() {
		return errorStr;
	}

	/** 
	 * Return the initialState of the transition system
	 */
	public S getInitial() {
		return initialState;
	}
	public boolean equals(Object o){
		if(! (o instanceof SimpleState))
			return false;
		if( ((SimpleState)o).getName().equals(name))
			return true;
		else return false;
	}
	@SuppressWarnings("unchecked")
	public Set<A> getActionsOfStateWithName(S from, String name)
	throws Exception {
	if (from==null)
		throw new Exception("the argument from is null");
	Iterator<Transition> itAction=directedGraph.outgoingEdgesOf(from).iterator();
	HashSet<Transition> result=new HashSet<Transition>();
	while(itAction.hasNext()){
		Transition currentAction=itAction.next();
		if(currentAction.getName().equals(name))
			result.add(currentAction);
	}
	return (Set<A>)result;

	}


}
